let currentPdf = null;
let currentFile = null;
let currentTool = 'text';
let currentShapeType = 'RECTANGLE';
let pendingImageData = null;
let selectedElement = null;
let currentZoom = 1.0;
const scale = 1.0; // Defined scale to fix ReferenceError

const actionHistory = [];
const redoStack = [];

// --- Initialization ---
document.getElementById('fileInput').addEventListener('change', async function (e) {
    if (this.files[0]) {
        currentFile = this.files[0];

        document.getElementById('step-upload').classList.add('hidden');
        document.getElementById("step-editor").classList.remove("hidden");

        try {
            const arrayBuffer = await currentFile.arrayBuffer();
            await loadPdf(arrayBuffer);
            showToast("Select a tool from the sidebar to start editing.");
        } catch (err) {
            console.error("Error loading PDF:", err);
            alert("Failed to load PDF. Please try another file.");
            document.getElementById('uploadOverlay').style.display = 'flex';
        }
    }
});


document.addEventListener('keydown', function (e) {
    if ((e.ctrlKey || e.metaKey) && e.key === 'z') {
        e.preventDefault();
        undo();
    }
    if ((e.ctrlKey || e.metaKey) && (e.key === 'y' || (e.shiftKey && e.key === 'Z'))) {
        e.preventDefault();
        redo();
    }
});

document.getElementById('imageInput').addEventListener('change', function (e) {
    if (this.files && this.files[0]) {
        const reader = new FileReader();
        reader.onload = function (evt) {
            pendingImageData = evt.target.result;
            currentTool = 'image_dropper';
            document.body.style.cursor = 'crosshair';
            showToast("Click anywhere on the page to place the image.");
        };
        reader.readAsDataURL(this.files[0]);
    }
    this.value = '';
});

async function loadPdf(data) {
    const loadingTask = pdfjsLib.getDocument(data);
    currentPdf = await loadingTask.promise;
    const container = document.getElementById('zoomWrapper');
    container.innerHTML = '';

    for (let pageNum = 1; pageNum <= currentPdf.numPages; pageNum++) {
        const page = await currentPdf.getPage(pageNum);
        const viewport = page.getViewport({scale: 1.0}); // Base scale 1.0

        const pageWrapper = document.createElement('div');
        pageWrapper.className = 'pdf-page-container';
        pageWrapper.style.width = `${viewport.width}px`;
        pageWrapper.style.height = `${viewport.height}px`;
        pageWrapper.dataset.pageNum = pageNum;
        pageWrapper.dataset.originalHeight = viewport.height;

        const canvas = document.createElement('canvas');
        canvas.className = 'pdf-canvas';
        canvas.height = viewport.height;
        canvas.width = viewport.width;

        const renderContext = {
            canvasContext: canvas.getContext('2d'),
            viewport: viewport
        };
        await page.render(renderContext).promise;

        const layer = document.createElement('div');
        layer.className = 'interaction-layer';

        // Unified event handling
        layer.addEventListener('mousedown', (e) => handleLayerMouseDown(e, layer, pageNum));

        pageWrapper.appendChild(canvas);
        pageWrapper.appendChild(layer);
        container.appendChild(pageWrapper);
    }
}

// --- Zoom Logic ---
function zoom(change) {
    currentZoom += change;
    if (currentZoom < 0.5) currentZoom = 0.5;
    if (currentZoom > 3.0) currentZoom = 3.0;

    // Use CSS zoom property instead of transform: scale
    document.getElementById('zoomWrapper').style.zoom = currentZoom;
    document.getElementById('zoomLevel').innerText = Math.round(currentZoom * 100) + '%';
}

function recordAction(action) {
    actionHistory.push(action);
    redoStack.length = 0;
}

function undo() {
    if (actionHistory.length === 0) return;
    const action = actionHistory.pop();
    redoStack.push(action);

    if (action.type === 'add') {
        action.element.remove();
        deselectElement();
    } else if (action.type === 'delete') {
        action.parent.appendChild(action.element);
    } else if (action.type === 'add_drawing') {
        action.elements.forEach(el => el.remove());
    } else if (action.type === 'delete_drawing') {
        action.elements.forEach(el => action.parent.appendChild(el));
    }
}

function redo() {
    if (redoStack.length === 0) return;
    const action = redoStack.pop();
    actionHistory.push(action);

    if (action.type === 'add') {
        action.parent.appendChild(action.element);
    } else if (action.type === 'delete') {
        action.element.remove();
        deselectElement();
    } else if (action.type === 'add_drawing') {
        action.elements.forEach(el => action.parent.appendChild(el));
    } else if (action.type === 'delete_drawing') {
        action.elements.forEach(el => el.remove());
    }
}

function setTool(tool) {
    currentTool = tool;
    pendingImageData = null;
    document.body.style.cursor = 'default';

    const shapeMenu = document.getElementById('shapeMenu');
    if (tool !== 'shape' && !shapeMenu.classList.contains('hidden')) {
        shapeMenu.classList.add('hidden');
    }

    deselectElement();

    document.querySelectorAll('.tool-btn').forEach(b => b.classList.remove('active'));

    // Hide all properties first
    document.getElementById('textProperties').classList.add('hidden');
    document.getElementById('shapeProperties').classList.add('hidden');
    document.getElementById('drawProperties').classList.add('hidden');

    if (tool === 'text') {
        document.getElementById('toolText').classList.add('active');
        document.getElementById('textProperties').classList.remove('hidden');
        showToast("Click anywhere to add text.");
    } else if (tool === 'whiteout') {
        document.getElementById('toolWhiteout').classList.add('active');
        showToast("Click and drag to create a whiteout box.");
    } else if (tool === 'shape') {
        document.getElementById('toolShape').classList.add('active');
        document.getElementById('shapeProperties').classList.remove('hidden');
        showToast("Click and drag to create a " + currentShapeType.toLowerCase() + ".");
    } else if (tool === 'draw') {
        document.getElementById('toolDraw').classList.add('active');
        document.getElementById('drawProperties').classList.remove('hidden');
        document.body.style.cursor = 'crosshair';
        showToast("Click and drag to draw.");
    }
}

function toggleShapeMenu() {
    const menu = document.getElementById('shapeMenu');
    menu.classList.toggle('hidden');
}

function selectShape(type) {
    currentShapeType = type;
    setTool('shape');
    document.getElementById('shapeMenu').classList.add('hidden');
}

function triggerImageUpload() {
    document.getElementById('imageInput').click();
}

function selectElement(el) {
    if (selectedElement) {
        selectedElement.classList.remove('selected');
        if (selectedElement.classList.contains('draggable-text')) {
            selectedElement.contentEditable = false;
        }
    }
    selectedElement = el;
    el.classList.add('selected');

    // Hide all properties first
    document.getElementById('textProperties').classList.add('hidden');
    document.getElementById('shapeProperties').classList.add('hidden');
    document.getElementById('drawProperties').classList.add('hidden');

    if (el.classList.contains('draggable-text')) {
        document.getElementById('textProperties').classList.remove('hidden');
        document.getElementById('textColor').value = rgbToHex(el.style.color);
        document.getElementById('textSize').value = el.dataset.fontSize;
        el.contentEditable = true;
        el.focus();
    } else if (el.classList.contains('shape-box')) {
        document.getElementById('shapeProperties').classList.remove('hidden');
        document.getElementById('shapeStroke').value = rgbToHex(el.dataset.stroke || el.style.borderColor);
        document.getElementById('shapeFill').value = rgbToHex(el.dataset.fill || el.style.backgroundColor);
    }
}

function deselectElement() {
    if (selectedElement) {
        selectedElement.classList.remove('selected');
        if (selectedElement.classList.contains('draggable-text')) {
            selectedElement.contentEditable = false;
        }
        selectedElement = null;
    }
}

// Unified Mouse Down Handler
function handleLayerMouseDown(e, layer, pageNum) {
    if (e.target !== layer) return;

    // Handle Click-to-Place Tools (Text, Image)
    if (currentTool === 'text' || (currentTool === 'image_dropper' && pendingImageData)) {
        // Treat as click
        deselectElement();
        const rect = layer.getBoundingClientRect();
        // When using CSS zoom, client coordinates are already scaled by the browser for the element,
        // but we need to be careful.
        // Actually, with CSS zoom, getBoundingClientRect returns the visual dimensions (scaled).
        // e.clientX/Y are viewport coordinates.
        // To get the internal coordinate relative to the unscaled element content:
        // We can use offsetX/offsetY if available and reliable, or calculate manually.
        
        // However, since we are using CSS zoom on the container, the layer itself is visually scaled.
        // If we append a child at (x,y), that child will also be scaled by the zoom.
        // So we just need the coordinate relative to the layer's top-left corner, 
        // BUT we need to account for the fact that the layer's visual size is different from its layout size if zoom is applied.
        
        // Wait, if we use CSS zoom on the wrapper, the children (pages) are scaled.
        // The coordinate system inside the page is also scaled?
        // No, if I set left: 100px, it will look like 100*zoom px.
        // So if I click at visual 100px (which is 50px logical if zoom is 2), I should set left to 50px.
        
        // Let's check how offsetX works with zoom.
        // Usually offsetX is correct relative to the target element's padding box.
        // Let's try using offsetX/offsetY which is much simpler if it works.
        
        // If offsetX is not reliable, we fall back to client rects.
        // rect.left is the visual left. e.clientX is visual x.
        // (e.clientX - rect.left) is the visual distance.
        // We need to divide by currentZoom to get the logical distance.
        
        const x = (e.clientX - rect.left) / currentZoom; 
        const y = (e.clientY - rect.top) / currentZoom;

        if (currentTool === 'text') {
            addTextElement(layer, x, y);
        } else if (currentTool === 'image_dropper') {
            addImageElement(layer, x, y, pendingImageData);
            setTool('text');
        }
        return;
    }

    // Handle Drag-to-Create Tools (Draw, Whiteout, Shape)
    if (currentTool === 'draw') {
        handleDrawing(e, layer);
        return;
    }

    if (currentTool === 'whiteout' || currentTool === 'shape') {
        handleShapeCreation(e, layer);
        return;
    }
}

// Drawing Logic
function handleDrawing(e, layer) {
    deselectElement();
    let isDrawing = true;

    const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
    svg.classList.add('drawing-box');
    svg.style.width = '100%';
    svg.style.height = '100%';

    const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
    const color = document.getElementById('drawColor').value;
    const width = document.getElementById('drawWidth').value;

    path.setAttribute("stroke", color);
    path.setAttribute("stroke-width", width);
    path.setAttribute("fill", "none");
    path.setAttribute("stroke-linecap", "round");
    path.setAttribute("stroke-linejoin", "round");

    svg.appendChild(path);
    layer.appendChild(svg);

    const startX = (e.clientX - layer.getBoundingClientRect().left) / currentZoom;
    const startY = (e.clientY - layer.getBoundingClientRect().top) / currentZoom;
    let points = [{x: startX, y: startY}];
    path.setAttribute("d", `M ${startX} ${startY}`);

    const drawMove = (ev) => {
        if (!isDrawing) return;
        const cx = (ev.clientX - layer.getBoundingClientRect().left) / currentZoom;
        const cy = (ev.clientY - layer.getBoundingClientRect().top) / currentZoom;
        points.push({x: cx, y: cy});
        const d = path.getAttribute("d");
        path.setAttribute("d", `${d} L ${cx} ${cy}`);
    };

    const drawUp = () => {
        isDrawing = false;
        svg.dataset.points = JSON.stringify(points);
        svg.dataset.color = color;
        svg.dataset.width = width;

        const lastPoint = points[points.length - 1];
        const delBtn = document.createElement('div');
        delBtn.className = 'delete-btn';
        delBtn.innerText = '×';
        delBtn.style.left = lastPoint.x + 'px';
        delBtn.style.top = lastPoint.y + 'px';

        delBtn.onclick = (ev) => {
            ev.stopPropagation();
            recordAction({type: 'delete_drawing', elements: [svg, delBtn], parent: layer});
            svg.remove();
            delBtn.remove();
        };

        layer.appendChild(delBtn);
        delBtn.classList.add('drawing-delete-btn');

        recordAction({type: 'add_drawing', elements: [svg, delBtn], parent: layer});

        window.removeEventListener('mousemove', drawMove);
        window.removeEventListener('mouseup', drawUp);
    };

    window.addEventListener('mousemove', drawMove);
    window.addEventListener('mouseup', drawUp);
}

// Shape/Whiteout Creation Logic
function handleShapeCreation(e, layer) {
    deselectElement();
    let isDrawing = true;
    const startX = (e.clientX - layer.getBoundingClientRect().left) / currentZoom;
    const startY = (e.clientY - layer.getBoundingClientRect().top) / currentZoom;

    const drawingRect = document.createElement('div');
    drawingRect.style.position = 'absolute';
    drawingRect.style.border = '1px dashed #667eea';
    drawingRect.style.backgroundColor = 'rgba(102, 126, 234, 0.2)';
    drawingRect.style.left = startX + 'px';
    drawingRect.style.top = startY + 'px';
    drawingRect.style.width = '0px';
    drawingRect.style.height = '0px';
    drawingRect.style.zIndex = '20';
    layer.appendChild(drawingRect);

    const mouseMoveHandler = (moveEvent) => {
        if (!isDrawing) return;
        const currentX = (moveEvent.clientX - layer.getBoundingClientRect().left) / currentZoom;
        const currentY = (moveEvent.clientY - layer.getBoundingClientRect().top) / currentZoom;

        const width = currentX - startX;
        const height = currentY - startY;

        drawingRect.style.left = (width < 0 ? currentX : startX) + 'px';
        drawingRect.style.top = (height < 0 ? currentY : startY) + 'px';
        drawingRect.style.width = Math.abs(width) + 'px';
        drawingRect.style.height = Math.abs(height) + 'px';
    };

    const mouseUpHandler = (upEvent) => {
        if (!isDrawing) return;
        isDrawing = false;
        layer.removeChild(drawingRect);

        const finalX = parseFloat(drawingRect.style.left);
        const finalY = parseFloat(drawingRect.style.top);
        const finalWidth = parseFloat(drawingRect.style.width);
        const finalHeight = parseFloat(drawingRect.style.height);

        if (finalWidth > 5 && finalHeight > 5) {
            if (currentTool === 'whiteout') {
                addWhiteoutElement(layer, finalX, finalY, finalWidth, finalHeight);
            } else if (currentTool === 'shape') {
                addShapeElement(layer, finalX, finalY, finalWidth, finalHeight, currentShapeType);
            }
        }

        window.removeEventListener('mousemove', mouseMoveHandler);
        window.removeEventListener('mouseup', mouseUpHandler);
    };

    window.addEventListener('mousemove', mouseMoveHandler);
    window.addEventListener('mouseup', mouseUpHandler);
}


// --- Elements ---
function addTextElement(layer, x, y) {
    const div = document.createElement('div');
    div.className = 'draggable-text';
    div.contentEditable = true;
    div.innerText = "Type here";
    div.style.left = x + 'px';
    div.style.top = y + 'px';

    const color = document.getElementById('textColor').value;
    const size = document.getElementById('textSize').value;
    div.style.color = color;
    div.style.fontSize = (size * scale) + 'px';
    div.dataset.fontSize = size;

    addCommonControls(div);
    makeDraggable(div);
    layer.appendChild(div);

    recordAction({type: 'add', element: div, parent: layer});

    selectElement(div);
    setTimeout(() => {
        div.focus();
        document.execCommand('selectAll', false, null);
    }, 0);
}

function addWhiteoutElement(layer, x, y, w = 100, h = 30) {
    const div = document.createElement('div');
    div.className = 'redaction-box';
    div.style.left = x + 'px';
    div.style.top = y + 'px';
    div.style.width = w + 'px';
    div.style.height = h + 'px';

    addCommonControls(div, true);
    makeDraggable(div);
    layer.appendChild(div);

    recordAction({type: 'add', element: div, parent: layer});
    selectElement(div);
}

function addImageElement(layer, x, y, src, w = 150, h = 150) {
    const div = document.createElement('div');
    div.className = 'image-box';
    div.style.left = x + 'px';
    div.style.top = y + 'px';
    div.style.width = w + 'px';
    div.style.height = h + 'px';

    const img = document.createElement('img');
    img.src = src;
    div.appendChild(img);

    addCommonControls(div, true);
    makeDraggable(div);
    layer.appendChild(div);

    recordAction({type: 'add', element: div, parent: layer});
    selectElement(div);
}

function addShapeElement(layer, x, y, w = 100, h = 100, type) {
    const div = document.createElement('div');
    div.className = 'shape-box';
    div.style.left = x + 'px';
    div.style.top = y + 'px';
    div.style.width = w + 'px';
    div.style.height = h + 'px';
    div.dataset.type = type;

    const stroke = document.getElementById('shapeStroke').value;
    const fill = document.getElementById('shapeFill').value;

    if (type === 'RECTANGLE') {
        div.style.border = `2px solid ${stroke}`;
        div.style.backgroundColor = fill;
    } else if (type === 'CIRCLE') {
        div.style.border = `2px solid ${stroke}`;
        div.style.backgroundColor = fill;
        div.style.borderRadius = '50%';
    } else if (type === 'LINE') {
        div.style.height = '2px';
        div.style.backgroundColor = stroke;
        div.style.border = 'none';
        div.style.transformOrigin = '0 0';
    } else if (type === 'TRIANGLE') {
        div.innerHTML = `<svg width="100%" height="100%" viewBox="0 0 100 100" preserveAspectRatio="none">
                    <polygon points="50,0 100,100 0,100" style="fill:${fill};stroke:${stroke};stroke-width:2" />
                </svg>`;
        div.style.border = 'none';
    } else if (type === 'STAR') {
        div.innerHTML = `<svg width="100%" height="100%" viewBox="0 0 100 100" preserveAspectRatio="none">
                    <polygon points="50,0 61,35 98,35 68,57 79,91 50,70 21,91 32,57 2,35 39,35" style="fill:${fill};stroke:${stroke};stroke-width:2" />
                </svg>`;
        div.style.border = 'none';
    }

    div.dataset.stroke = stroke;
    div.dataset.fill = fill;

    addCommonControls(div, true);
    makeDraggable(div);
    layer.appendChild(div);

    recordAction({type: 'add', element: div, parent: layer});
    selectElement(div);
}

function addCommonControls(el, resizable = false) {
    const delBtn = document.createElement('div');
    delBtn.className = 'delete-btn';
    delBtn.innerText = '×';
    delBtn.contentEditable = false;
    delBtn.onclick = (e) => {
        e.stopPropagation();
        recordAction({type: 'delete', element: el, parent: el.parentNode});
        el.remove();
        deselectElement();
    };
    el.appendChild(delBtn);

    el.addEventListener('mousedown', (e) => {
        e.stopPropagation();
        selectElement(el);
    });

    if (resizable) {
        const resizeHandle = document.createElement('div');
        resizeHandle.className = 'resize-handle';
        el.appendChild(resizeHandle);
        makeResizable(el, resizeHandle);
    }
}

function makeDraggable(el) {
    let isDragging = false;
    let startX, startY, initialLeft, initialTop;

    el.addEventListener('mousedown', (e) => {
        if (e.target.classList.contains('delete-btn') || e.target.classList.contains('resize-handle')) return;

        isDragging = true;
        startX = e.clientX;
        startY = e.clientY;
        initialLeft = parseFloat(el.style.left);
        initialTop = parseFloat(el.style.top);
        el.style.cursor = 'grabbing';
        e.stopPropagation();
    });

    window.addEventListener('mousemove', (e) => {
        if (!isDragging) return;
        e.preventDefault();
        const dx = (e.clientX - startX) / currentZoom;
        const dy = (e.clientY - startY) / currentZoom;
        el.style.left = (initialLeft + dx) + 'px';
        el.style.top = (initialTop + dy) + 'px';
    });

    window.addEventListener('mouseup', () => {
        if (isDragging) {
            isDragging = false;
            el.style.cursor = 'move';
        }
    });
}

function makeResizable(el, handle) {
    let isResizing = false;
    let startX, startY, initialWidth, initialHeight;

    handle.addEventListener('mousedown', (e) => {
        isResizing = true;
        startX = e.clientX;
        startY = e.clientY;
        initialWidth = parseFloat(getComputedStyle(el).width);
        initialHeight = parseFloat(getComputedStyle(el).height);
        e.stopPropagation();
    });

    window.addEventListener('mousemove', (e) => {
        if (!isResizing) return;
        e.preventDefault();
        const dx = (e.clientX - startX) / currentZoom;
        const dy = (e.clientY - startY) / currentZoom;
        el.style.width = Math.max(20, initialWidth + dx) + 'px';
        el.style.height = Math.max(20, initialHeight + dy) + 'px';
    });

    window.addEventListener('mouseup', () => {
        isResizing = false;
    });
}

function clearAllElements() {
    if (confirm("Are you sure you want to clear all added elements?")) {
        document.querySelectorAll('.draggable-text, .redaction-box, .image-box, .shape-box, .drawing-box, .drawing-delete-btn').forEach(el => el.remove());
        deselectElement();
        showToast("All elements cleared.");
    }
}

async function savePdf() {
    if (!currentFile) return;

    document.getElementById('loadingOverlay').classList.remove('hidden');

    const textAdditions = [];
    const textRemovals = [];
    const imageAdditions = [];
    const shapeAdditions = [];
    const drawingAdditions = [];

    document.querySelectorAll('.pdf-page-container').forEach(pageWrapper => {
        const pageNum = parseInt(pageWrapper.dataset.pageNum);
        const pageHeight = parseFloat(pageWrapper.dataset.originalHeight); // Use original height

        pageWrapper.querySelectorAll('.draggable-text').forEach(el => {
            const text = el.innerText.replace('×', '').trim();
            if (!text) return;
            const rect = el.getBoundingClientRect();
            const parentRect = pageWrapper.getBoundingClientRect();
            const xDom = rect.left - parentRect.left;
            const yDom = rect.top - parentRect.top;

            textAdditions.push({
                text: text,
                pageNumber: pageNum,
                x: xDom / currentZoom, // Adjust for zoom
                y: (pageHeight) - (yDom / currentZoom) - parseInt(el.dataset.fontSize),
                fontSize: parseInt(el.dataset.fontSize),
                color: rgbToHex(el.style.color)
            });
        });

        pageWrapper.querySelectorAll('.redaction-box').forEach(el => {
            const rect = el.getBoundingClientRect();
            const parentRect = pageWrapper.getBoundingClientRect();
            const xDom = rect.left - parentRect.left;
            const yDom = rect.top - parentRect.top;
            const hDom = rect.height;

            textRemovals.push({
                pageNumber: pageNum,
                x: xDom / currentZoom,
                y: (pageHeight) - (yDom / currentZoom) - (hDom / currentZoom),
                width: rect.width / currentZoom,
                height: hDom / currentZoom
            });
        });

        pageWrapper.querySelectorAll('.image-box').forEach(el => {
            const rect = el.getBoundingClientRect();
            const parentRect = pageWrapper.getBoundingClientRect();
            const xDom = rect.left - parentRect.left;
            const yDom = rect.top - parentRect.top;
            const hDom = rect.height;
            const img = el.querySelector('img');

            imageAdditions.push({
                base64Data: img.src,
                pageNumber: pageNum,
                x: xDom / currentZoom,
                y: (pageHeight) - (yDom / currentZoom) - (hDom / currentZoom),
                width: rect.width / currentZoom,
                height: hDom / currentZoom
            });
        });

        pageWrapper.querySelectorAll('.shape-box').forEach(el => {
            const rect = el.getBoundingClientRect();
            const parentRect = pageWrapper.getBoundingClientRect();
            const xDom = rect.left - parentRect.left;
            const yDom = rect.top - parentRect.top;
            const hDom = rect.height;
            const type = el.dataset.type || 'RECTANGLE';

            shapeAdditions.push({
                type: type,
                pageNumber: pageNum,
                x: xDom / currentZoom,
                y: (pageHeight) - (yDom / currentZoom) - (hDom / currentZoom),
                width: rect.width / currentZoom,
                height: hDom / currentZoom,
                strokeColor: rgbToHex(el.dataset.stroke || el.style.borderColor),
                fillColor: rgbToHex(el.dataset.fill || el.style.backgroundColor),
                lineWidth: 2.0
            });
        });

        pageWrapper.querySelectorAll('.drawing-box').forEach(el => {
            const points = JSON.parse(el.dataset.points || '[]');
            if (points.length === 0) return;

            const pdfPoints = points.map(p => ({
                x: p.x,
                y: (pageHeight) - p.y
            }));

            drawingAdditions.push({
                pageNumber: pageNum,
                points: pdfPoints,
                color: rgbToHex(el.dataset.color),
                lineWidth: parseFloat(el.dataset.width)
            });
        });
    });

    const requestData = {
        textAdditions: textAdditions,
        textRemovals: textRemovals,
        imageAdditions: imageAdditions,
        shapeAdditions: shapeAdditions,
        drawingAdditions: drawingAdditions
    };

    const formData = new FormData();
    formData.append('file', currentFile);
    formData.append('editRequest', JSON.stringify(requestData));

    try {
        const response = await fetch('/edit-pdf-content', {
            method: 'POST',
            body: formData
        });

        if (response.ok) {
            const blob = await response.blob();
            const url = window.URL.createObjectURL(blob);
            
            // Setup download button
            const downloadBtn = document.getElementById("downloadBtn");
            downloadBtn.href = url;
            downloadBtn.download = "edited_" + currentFile.name;

            // Switch screen
            document.getElementById("step-editor").classList.add("hidden");
            document.getElementById("step-success").classList.remove("hidden");

        } else {
            alert('Error processing PDF');
        }
    } catch (error) {
        console.error(error);
        alert('An error occurred');
    } finally {
        document.getElementById('loadingOverlay').classList.add('hidden');
    }
}

async function sharePdf() {
    if (navigator.share) {
        try {
            await navigator.share({
                title: 'PDF Nex - Edit PDF',
                text: 'Check out this PDF editor!',
                url: window.location.href
            });
        } catch (err) {
            console.error('Share failed:', err);
        }
    } else {
        alert('Sharing is not supported on this browser. Copy this link: ' + window.location.href);
    }
}

const rgbToHex = (rgb) => {
    if (!rgb || rgb.startsWith('#') || rgb === 'transparent') return rgb || "#000000";
    const [r, g, b] = rgb.match(/\d+/g);
    return "#" + ((1 << 24) + (parseInt(r) << 16) + (parseInt(g) << 8) + parseInt(b)).toString(16).slice(1);
};
