/*************************************************
 * SIGN PDF – GRID BASED (FIXED & STABLE)
 *************************************************/

pdfjsLib.GlobalWorkerOptions.workerSrc =
  "https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js";

/* ---------------- STATE ---------------- */
let pdfFile, pdfJsDoc, pdfDoc;
let signatureDataUrl = null;
let signatureImageBytes = null;
let activeSignatureId = null;

const signatures = []; 
// { id, pageIndex, x, y, w, h, dataUrl, bytes, el }

/* ---------------- DOM ---------------- */
const fileInput = document.getElementById("fileInput");
const dropZone = document.getElementById("dropZone");
const stepUpload = document.getElementById("step-upload");
const stepEditor = document.getElementById("step-editor");
const stepSuccess = document.getElementById("step-success");
const loadingOverlay = document.getElementById("loadingOverlay");
const gridContent = document.getElementById("gridContent");
const downloadBtn = document.getElementById("downloadBtn");
const applyBtn = document.getElementById("applyBtn");
const selectedPageInfo = document.getElementById("selectedPageInfo");

/* ---------------- UPLOAD ---------------- */
dropZone.onclick = () => fileInput.click();

fileInput.onchange = async (e) => {
  if (!e.target.files.length) return;

  pdfFile = e.target.files[0];
  loadingOverlay.classList.remove("hidden");

  const buffer = await pdfFile.arrayBuffer();
  pdfJsDoc = await pdfjsLib.getDocument({ data: buffer }).promise;

  stepUpload.classList.add("hidden");
  stepEditor.classList.remove("hidden");

  await renderPages();
  loadingOverlay.classList.add("hidden");
};

/* ---------------- RENDER ALL PAGES ---------------- */
async function renderPages() {
  gridContent.innerHTML = "";

  for (let i = 0; i < pdfJsDoc.numPages; i++) {
    const page = await pdfJsDoc.getPage(i + 1);
    const viewport = page.getViewport({ scale: 1 });

    const canvas = document.createElement("canvas");
    const ctx = canvas.getContext("2d");

    canvas.width = viewport.width;
    canvas.height = viewport.height;
    canvas.className = "bg-white shadow border rounded cursor-pointer";

    await page.render({ canvasContext: ctx, viewport }).promise;

    canvas.onclick = (e) => placeSignature(e, canvas, i);

    const wrapper = document.createElement("div");
    wrapper.style.position = "relative";
    wrapper.appendChild(canvas);

    gridContent.appendChild(wrapper);
  }
}

/* ---------------- PLACE SIGNATURE ---------------- */
function placeSignature(e, canvas, pageIndex) {
  if (!signatureDataUrl || !signatureImageBytes) {
    alert("Save signature first");
    return;
  }

  const rect = canvas.getBoundingClientRect();
  const x = e.clientX - rect.left;
  const y = canvas.height - (e.clientY - rect.top);

  const sig = {
    id: crypto.randomUUID(),
    pageIndex,
    x,
    y,
    w: 150,
    h: 60,
    dataUrl: signatureDataUrl,
    bytes: signatureImageBytes,
    el: null
  };

  signatures.push(sig);
  drawSignature(canvas, sig);
}

/* ---------------- DRAW PREVIEW ---------------- */
function drawSignature(canvas, sig) {
  const wrapper = canvas.parentElement;

  const el = document.createElement("div");
  el.className = "signature-overlay";
  el.style.cssText = `
    position:absolute;
    left:${sig.x}px;
    top:${canvas.height - sig.y}px;
    width:${sig.w}px;
    height:${sig.h}px;
    cursor:move;
    z-index:10;
  `;
  el.style.opacity = signatureOptions.opacity;
  el.style.transform = `rotate(${signatureOptions.rotation}deg)`;
  
  const img = document.createElement("img");
  img.src = sig.dataUrl;
  img.style.width = "100%";
  img.style.height = "100%";
  img.draggable = false;

  el.appendChild(img);
  wrapper.appendChild(el);
  sig.el = el;

  enableDrag(el, canvas, sig);
  el.onclick = () => selectSignature(sig.id);
}

/* ---------------- DRAG ---------------- */
function enableDrag(el, canvas, sig) {
  let sx, sy;

  el.onmousedown = (e) => {
    sx = e.clientX;
    sy = e.clientY;

    const move = (ev) => {
      const dx = ev.clientX - sx;
      const dy = ev.clientY - sy;

      el.style.left = el.offsetLeft + dx + "px";
      el.style.top = el.offsetTop + dy + "px";

      sig.x = el.offsetLeft;
      sig.y = canvas.height - el.offsetTop;

      sx = ev.clientX;
      sy = ev.clientY;
    };

    document.addEventListener("mousemove", move);
    document.addEventListener("mouseup", () =>
      document.removeEventListener("mousemove", move),
      { once: true }
    );
  };
}

/* ---------------- SELECT / DELETE ---------------- */
function selectSignature(id) {
  activeSignatureId = id;
  signatures.forEach(s => {
    if (s.el)
      s.el.style.outline = s.id === id ? "2px solid #4f46e5" : "none";
  });

  const sig = signatures.find(s => s.id === id);
  selectedPageInfo.innerText = sig ? "Page " + (sig.pageIndex + 1) : "None";
}

window.deleteSelectedSignature = () => {
  if (!activeSignatureId) return;
  const i = signatures.findIndex(s => s.id === activeSignatureId);
  if (i === -1) return;
  signatures[i].el.remove();
  signatures.splice(i, 1);
  activeSignatureId = null;
  selectedPageInfo.innerText = "None";
};

/* ---------------- SIGNATURE CREATION ---------------- */
window.saveSignature = async () => {
  const typeInput = document.getElementById("typedSignature");
  const drawCanvas = document.getElementById("signatureCanvas");

  // ---- TYPE SIGNATURE ----
  if (!typeInput.classList.contains("hidden") && typeInput.value.trim()) {
    const c = document.createElement("canvas");
    c.width = 600;
    c.height = 160;

    const ctx = c.getContext("2d");
    ctx.font = `${signatureOptions.fontSize}px ${signatureOptions.font}`;
    ctx.fillStyle = signatureOptions.color;
    ctx.globalAlpha = signatureOptions.opacity;
    ctx.fillText(typeInput.value.trim(), 20, 100);

    signatureDataUrl = c.toDataURL("image/png");
    signatureImageBytes = await fetch(signatureDataUrl).then(r => r.arrayBuffer());
  }

  // ---- DRAW SIGNATURE ----
  else if (!drawCanvas.classList.contains("hidden")) {
    signatureDataUrl = drawCanvas.toDataURL("image/png");
    signatureImageBytes = await fetch(signatureDataUrl).then(r => r.arrayBuffer());
  }

  else {
    alert("Create signature first");
    return;
  }

  alert("Signature saved. Click on PDF page.");
};


/* ---------------- APPLY PDF ---------------- */
applyBtn.onclick = async () => {
  if (!signatures.length) return alert("No signatures placed");

  loadingOverlay.classList.remove("hidden");

  const bytes = await pdfFile.arrayBuffer();
  pdfDoc = await PDFLib.PDFDocument.load(bytes);
  const pages = pdfDoc.getPages();

  for (const s of signatures) {
    const page = pages[s.pageIndex];
    const img = await pdfDoc.embedPng(s.bytes);

    page.drawImage(img, {
      x: s.x,
      y: s.y,
      width: s.w,
      height: s.h,
      rotate: PDFLib.degrees(signatureOptions.rotation),
      opacity: signatureOptions.opacity
    });
    
  }

  const out = await pdfDoc.save();
  const blob = new Blob([out], { type: "application/pdf" });
  const url = URL.createObjectURL(blob);

  downloadBtn.href = url;
  downloadBtn.download = "signed.pdf";

  stepEditor.classList.add("hidden");
  stepSuccess.classList.remove("hidden");
  loadingOverlay.classList.add("hidden");
};
/* ================= SIGNATURE MODE HANDLERS ================= */

window.openTypeSignature = () => {
  // show text input
  document.getElementById("typedSignature").classList.remove("hidden");
  document.getElementById("signatureCanvas").classList.add("hidden");
};

window.openDrawSignature = () => {
  const canvas = document.getElementById("signatureCanvas");
  const typeInput = document.getElementById("typedSignature");

  canvas.classList.remove("hidden");
  typeInput.classList.add("hidden");

  const ctx = canvas.getContext("2d");
  canvas.width = canvas.offsetWidth;
  canvas.height = canvas.offsetHeight;

  ctx.clearRect(0, 0, canvas.width, canvas.height);
  ctx.lineCap = "round";

  let drawing = false;

  canvas.onmousedown = () => {
    drawing = true;
    ctx.beginPath();
  };

  canvas.onmouseup = () => {
    drawing = false;
    ctx.beginPath();
  };

  canvas.onmousemove = (e) => {
    if (!drawing) return;
    ctx.strokeStyle = signatureOptions.color;          // ✅ color
    ctx.lineWidth = signatureOptions.drawThickness;    // ✅ thickness
    ctx.lineTo(e.offsetX, e.offsetY);
    ctx.stroke();
    ctx.beginPath();
    ctx.moveTo(e.offsetX, e.offsetY);
  };
};


window.openUploadSignature = () => {
  document.getElementById("signatureImageInput").click();
};
/* ================= DATE SIGNATURE ================= */

window.insertDateSignature = async () => {
  const date = new Date().toLocaleDateString();

  const c = document.createElement("canvas");
  c.width = 500;
  c.height = 120;

  const ctx = c.getContext("2d");
  ctx.font = `${signatureOptions.fontSize}px ${signatureOptions.font}`;
  ctx.fillStyle = signatureOptions.color;
  ctx.fillText(date, 20, 70);

  signatureDataUrl = c.toDataURL("image/png");
  signatureImageBytes = await fetch(signatureDataUrl).then(r => r.arrayBuffer());

  alert("Date signature ready. Click on PDF page.");
};
/* ================= SIGNATURE OPTIONS ================= */

window.signatureOptions = {
  color: "#000000",
  font: "Pacifico",
  fontSize: 48,
  opacity: 1,
  rotation: 0, // degrees
  drawThickness: 2
};

